<?php
namespace ArabesqueCore\CPT\Shortcodes\Separator;

use ArabesqueCore\Lib;

class Separator implements Lib\ShortcodeInterface {
	private $base;
	
	function __construct() {
		$this->base = 'mkdf_separator';
		add_action( 'vc_before_init', array( $this, 'vcMap' ) );
	}
	
	public function getBase() {
		return $this->base;
	}
	
	public function vcMap() {
		if ( function_exists( 'vc_map' ) ) {
			vc_map(
				array(
					'name'                    => esc_html__( 'Separator', 'arabesque-core' ),
					'base'                    => $this->base,
					'category'                => esc_html__( 'By ARABESQUE', 'arabesque-core' ),
					'icon'                    => 'icon-wpb-separator extended-custom-icon',
					'show_settings_on_create' => true,
					'class'                   => 'wpb_vc_separator',
					'custom_markup'           => '<div></div>',
					'params'                  => array_merge(
						array(
							array(
								'type'        => 'textfield',
								'param_name'  => 'custom_class',
								'heading'     => esc_html__( 'Custom CSS Class', 'arabesque-core' ),
								'description' => esc_html__( 'Style particular content element differently - add a class name and refer to it in custom CSS', 'arabesque-core' )
							),
							array(
								'type'       => 'dropdown',
								'param_name' => 'type',
								'heading'    => esc_html__( 'Type', 'arabesque-core' ),
								'value'      => array(
									esc_html__( 'Normal', 'arabesque-core' )     => 'normal',
									esc_html__( 'Full Width', 'arabesque-core' ) => 'full-width'
								)
							),
							array(
								'type'       => 'dropdown',
								'param_name' => 'position',
								'heading'    => esc_html__( 'Position', 'arabesque-core' ),
								'value'      => array(
									esc_html__( 'Center', 'arabesque-core' ) => 'center',
									esc_html__( 'Left', 'arabesque-core' )   => 'left',
									esc_html__( 'Right', 'arabesque-core' )  => 'right'
								),
								'dependency' => array( 'element' => 'type', 'value' => array( 'normal' ) )
							),
							array(
								'type'       => 'dropdown',
								'param_name' => 'enable_icon',
								'heading'    => esc_html__( 'Enable Icon', 'arabesque-core' ),
								'value'      => array(
									esc_html__( 'None', 'arabesque-core' )     => 'no',
									esc_html__( 'Icon from Icon Pack', 'arabesque-core' ) => 'icon-pack',
									esc_html__( 'Custom Icon', 'arabesque-core' ) => 'custom-icon',
								),
								'dependency' => array( 'element' => 'type', 'value' => array( 'normal' ) )
							),
						),
						arabesque_mikado_icon_collections()->getVCParamsArray(array('element' => 'enable_icon', 'value' => 'icon-pack'), '', true),
						array(
							array(
								'type'       => 'textfield',
								'param_name' => 'icon_size',
								'heading'    => esc_html__( 'Icon Size (px)', 'arabesque-core' ),
								'dependency' => array( 'element' => 'enable_icon', 'value' => array( 'icon-pack' ) )
							),
							array(
								'type'       => 'colorpicker',
								'param_name' => 'icon_color',
								'heading'    => esc_html__( 'Icon Color', 'arabesque-core' ),
								'dependency' => array( 'element' => 'enable_icon', 'value' => array( 'icon-pack' ) )
							),
							array(
								'type'       => 'attach_image',
								'param_name' => 'custom_icon',
								'heading'    => esc_html__( 'Icon', 'arabesque-core' ),
								'dependency' => array( 'element' => 'enable_icon', 'value' => array( 'custom-icon' ) )
							),
							array(
								'type'       => 'dropdown',
								'param_name' => 'icon_position',
								'heading'    => esc_html__( 'Icon Position', 'arabesque-core' ),
								'value'      => array(
									esc_html__( 'Center', 'arabesque-core' ) => 'center',
									esc_html__( 'Left', 'arabesque-core' )   => 'left',
									esc_html__( 'Right', 'arabesque-core' )  => 'right'
								),
								'dependency' => array( 'element' => 'enable_icon', 'value' => array( 'icon-pack','custom-icon' ) )
							),
							array(
								'type'       => 'textfield',
								'param_name' => 'icon_margin',
								'heading'    => esc_html__( 'Icon Left/Right Margin (px)', 'arabesque-core' ),
								'dependency' => array( 'element' => 'enable_icon', 'value' => array( 'icon-pack', 'custom-icon' ) )
							),
							array(
								'type'       => 'colorpicker',
								'param_name' => 'color',
								'heading'    => esc_html__( 'Color', 'arabesque-core' )
							),
                            array(
                                'type'       => 'dropdown',
                                'param_name' => 'animation',
                                'heading'    => esc_html__( 'Appear Animation', 'arabesque-core' ),
                                'value'      => array_flip( arabesque_mikado_get_yes_no_select_array( false, false ) ),
                                'dependency' => array( 'element' => 'type', 'value' => array( 'normal' ) )
                            ),
							array(
								'type'        => 'dropdown',
								'param_name'  => 'border_style',
								'heading'     => esc_html__( 'Style', 'arabesque-core' ),
								'value'       => array(
									esc_html__( 'Default', 'arabesque-core' ) => '',
									esc_html__( 'Dashed', 'arabesque-core' )  => 'dashed',
									esc_html__( 'Solid', 'arabesque-core' )   => 'solid',
									esc_html__( 'Dotted', 'arabesque-core' )  => 'dotted'
								),
								'save_always' => true
							),
							array(
								'type'       => 'textfield',
								'param_name' => 'width',
								'heading'    => esc_html__( 'Width (px or %)', 'arabesque-core' ),
								'dependency' => array( 'element' => 'type', 'value' => array( 'normal' ) )
							),
							array(
								'type'       => 'textfield',
								'param_name' => 'thickness',
								'heading'    => esc_html__( 'Thickness (px)', 'arabesque-core' )
							),
							array(
								'type'       => 'textfield',
								'param_name' => 'top_margin',
								'heading'    => esc_html__( 'Top Margin (px or %)', 'arabesque-core' )
							),
							array(
								'type'       => 'textfield',
								'param_name' => 'bottom_margin',
								'heading'    => esc_html__( 'Bottom Margin (px or %)', 'arabesque-core' )
							)
						)
					)
				)
			);
		}
	}
	
	public function render( $atts, $content = null ) {
		$args   = array(
			'custom_class'  => '',
			'type'          => '',
			'position'      => 'center',
			'enable_icon'	=> 'none',
			'custom_icon'	=> '',
			'icon_size'		=> '',
			'icon_color'	=> '',
			'icon_position' => 'center',
			'icon_margin'	=> '',
			'color'         => '',
            'animation'     => '',
			'border_style'  => '',
			'width'         => '',
			'thickness'     => '',
			'top_margin'    => '',
			'bottom_margin' => ''
		);
		$args = array_merge( $args, arabesque_mikado_icon_collections()->getShortcodeParams() );
		$params       = shortcode_atts( $args, $atts );
		
		$params['holder_classes'] = $this->getHolderClasses( $params );
		$params['holder_styles']  = $this->getHolderStyles( $params );
		$params['separator_styles']  = $this->getSeparatorStyles( $params );
		$params['icon_holder_style']  = $this->getIconHolderStyle( $params );
		$params['icon_html']  = $this->getIconHTML( $params );

		$html = arabesque_core_get_shortcode_module_template_part( 'templates/separator-template', 'separator', '', $params );
		
		return $html;
	}
	
	private function getHolderClasses( $params ) {
		$holderClasses = array();
		
		$holderClasses[] = ! empty( $params['custom_class'] ) ? esc_attr( $params['custom_class'] ) : '';
		$holderClasses[] = ! empty( $params['position'] ) ? 'mkdf-separator-' . $params['position'] : '';
		$holderClasses[] = ! empty( $params['type'] ) ? 'mkdf-separator-' . $params['type'] : '';
        $holderClasses[] = ! empty( $params['animation'] ) ? 'mkdf-separator-animation-' . $params['animation'] : '';
		
		return implode( ' ', $holderClasses );
	}
	
	private function getHolderStyles( $params ) {
		$styles = array();
		
		if ( $params['top_margin'] !== '' ) {
			if ( arabesque_mikado_string_ends_with( $params['top_margin'], '%' ) || arabesque_mikado_string_ends_with( $params['top_margin'], 'px' ) ) {
				$styles[] = 'margin-top: ' . $params['top_margin'];
			} else {
				$styles[] = 'margin-top: ' . arabesque_mikado_filter_px( $params['top_margin'] ) . 'px';
			}
		}
		
		if ( $params['bottom_margin'] !== '' ) {
			if ( arabesque_mikado_string_ends_with( $params['bottom_margin'], '%' ) || arabesque_mikado_string_ends_with( $params['bottom_margin'], 'px' ) ) {
				$styles[] = 'margin-bottom: ' . $params['bottom_margin'];
			} else {
				$styles[] = 'margin-bottom: ' . arabesque_mikado_filter_px( $params['bottom_margin'] ) . 'px';
			}
		}
		
		return implode( ';', $styles );
	}

	private function getSeparatorStyles( $params ) {
		$styles = array();
		
		if ( $params['color'] !== '' ) {
			$styles[] = 'border-color: ' . $params['color'];
		}
		
		if ( $params['border_style'] !== '' ) {
			$styles[] = 'border-style: ' . $params['border_style'];
		}
		
		if ( $params['width'] !== '' ) {
			$width_suffix = 'px';

			if ( arabesque_mikado_string_ends_with( $params['width'], '%' ) ) {
				$width_suffix = '%';
			}

			if ( $params['enable_icon'] !== 'none' && $params['icon_position'] == 'center') {
				$width = intval($params['width'])/2;
			} else {
				$width = intval($params['width']);
			}

			$styles[] = 'width: ' . $width . $width_suffix;
		}
		
		if ( $params['thickness'] !== '' ) {
			$styles[] = 'border-bottom-width: ' . arabesque_mikado_filter_px( $params['thickness'] ) . 'px';
		}
		
		return implode( ';', $styles );
	}

	private function getIconHolderStyle( $params ) {
		$holder_style = array();

		if ($params['icon_margin'] !== '') {
			if ($params['icon_position'] == 'center') {
				$holder_style[] = 'margin: 0 '.arabesque_mikado_filter_px($params['icon_margin']).'px';
			} elseif ($params['icon_position'] == 'left') {
				$holder_style[] = 'margin-right: '.arabesque_mikado_filter_px($params['icon_margin']).'px';
			} else {
				$holder_style[] = 'margin-left: '.arabesque_mikado_filter_px($params['icon_margin']).'px';
			}
		}

		return implode('; ', $holder_style);
	}

	private function getIconHTML( $params ) {
		$icon = '';

		if ( $params['enable_icon'] == 'custom-icon' && $params['custom_icon'] !== '' ) {

			$icon = wp_get_attachment_image($params['custom_icon'],'full');

		} elseif ( $params['enable_icon'] == 'icon-pack' ) {

			$iconPackName = arabesque_mikado_icon_collections()->getIconCollectionParamNameByKey( $params['icon_pack'] );
			
			$icon_parameters['icon_pack']     = $params['icon_pack'];
			$icon_parameters[ $iconPackName ] = $params[ $iconPackName ];
			
			if ( $params['icon_size'] !== '' ) {
				$icon_parameters['custom_size'] = arabesque_mikado_filter_px( $params['icon_size'] ) . 'px';
			}

			if ( $params['icon_color'] !== '' ) {
				$icon_parameters['icon_color'] = $params['icon_color'];
			}
			
			$icon = arabesque_mikado_execute_shortcode('mkdf_icon', $icon_parameters);
		}
		
		return $icon;
	}
}
