<?php
if (!defined('ABSPATH')) {
	exit;//Exit if accessed directly
}

class AIOWPSecurity_User_Registration {

	public function __construct() {
		global $aio_wp_security;
		add_action('user_register', array($this, 'aiowps_user_registration_action_handler'));

		if ($aio_wp_security->configs->get_value('aiowps_enable_manual_registration_approval') == '1') {
			add_filter("woocommerce_registration_auth_new_customer", array($this, 'aios_registration_auth_new_customer'));
		}

		if ($aio_wp_security->configs->get_value('aiowps_enable_registration_page_captcha') == '1') {
			add_filter('registration_errors', array($this, 'aiowps_validate_registration_with_captcha'), 10, 3);
		}
	}
	
	/**
	 * This public function will add a special meta string in the users table
	 * Meta field name: 'aiowps_account_status'
	 * Meta field value: 'pending'
	 *
	 * @param int $user_id
	 * @return void
	 */
	public function aiowps_user_registration_action_handler($user_id) {
		global $aio_wp_security;
		//Check if auto pending new account status feature is enabled
		if ($aio_wp_security->configs->get_value('aiowps_enable_manual_registration_approval') == '1') {
			if (AIOWPSecurity_Utility_Permissions::has_manage_cap() || (defined('WP_CLI') && WP_CLI)) {
				AIOWPSecurity_Audit_Events::event_user_registration($user_id, 'admin');
				return; //if the user has been added from admin side don't put in pending state
			}
			$res = add_user_meta($user_id, 'aiowps_account_status', 'pending');
			if (!$res) {
				$aio_wp_security->debug_logger->log_debug("aiowps_user_registration_action_handler: Error adding user meta data: aiowps_account_status", 4);
			}
			$user_ip_address = AIOWPSecurity_Utility_IP::get_user_ip_address();
			$res = add_user_meta($user_id, 'aiowps_registrant_ip', $user_ip_address);
			if (!$res) {
				$aio_wp_security->debug_logger->log_debug("aiowps_user_registration_action_handler: Error adding user meta data: aiowps_registrant_ip", 4);
			}
			AIOWPSecurity_Audit_Events::event_user_registration($user_id, 'pending');
		} else {
			if (AIOWPSecurity_Utility_Permissions::has_manage_cap()) {
				AIOWPSecurity_Audit_Events::event_user_registration($user_id, 'admin');
			} else {
				AIOWPSecurity_Audit_Events::event_user_registration($user_id, 'registered');
			}
		}
	}

	/**
	 * This public function will set the special meta string in the usermeta table so that the account becomes active
	 * Meta field name: 'aiowps_account_status'
	 * Meta field values: 'active', 'pending', etc
	 *
	 * @param int    $user_id
	 * @param string $status
	 * @return void
	 */
	public function aiowps_set_user_account_status($user_id, $status) {
		global $aio_wp_security;
		$res = update_user_meta($user_id, 'aiowps_account_status', $status);
		if (!$res) {
			$aio_wp_security->debug_logger->log_debug("aiowps_set_user_account_status: Error updating user meta data: aiowps_account_status", 4);
		}
	}
	
	public function aiowps_validate_registration_with_captcha($errors) {
		global $aio_wp_security;

		$locked = $aio_wp_security->user_login_obj->check_locked_user();
		if (null == $locked) {
			//user is not locked continue
		} else {
			$errors->add('authentication_failed', __('<strong>ERROR</strong>: You are not allowed to register because your IP address is currently locked!', 'all-in-one-wp-security-and-firewall'));
			return $errors;
		}
		$verify_captcha = $aio_wp_security->captcha_obj->verify_captcha_submit();
		if (false  === $verify_captcha) {
			// wrong answer was entered
			$errors->add('authentication_failed', __('<strong>ERROR</strong>: Your answer was incorrect - please try again.', 'all-in-one-wp-security-and-firewall'));
			return $errors;
		}
		return $errors;
	}

	/**
	 * This function serves the purpose of preventing login in certain plugins that enable user registration, such as WooCommerce and others.
	 *
	 * @return bool Retruns false means do not authenticate on registration
	 */
	public function aios_registration_auth_new_customer() {
		return false;
	}
}
