<?php
namespace ArabesqueCore\CPT\Shortcodes\ImageWithText;

use ArabesqueCore\Lib;

class ImageWithText implements Lib\ShortcodeInterface {
	private $base;
	
	public function __construct() {
		$this->base = 'mkdf_image_with_text';
		
		add_action( 'vc_before_init', array( $this, 'vcMap' ) );
	}
	
	public function getBase() {
		return $this->base;
	}
	
	public function vcMap() {
		if ( function_exists( 'vc_map' ) ) {
			vc_map(
				array(
					'name'                      => esc_html__( 'Image With Text', 'arabesque-core' ),
					'base'                      => $this->getBase(),
					'category'                  => esc_html__( 'By ARABESQUE', 'arabesque-core' ),
					'icon'                      => 'icon-wpb-image-with-text extended-custom-icon',
					'allowed_container_element' => 'vc_row',
					'params'                    => array(
						array(
							'type'        => 'textfield',
							'param_name'  => 'custom_class',
							'heading'     => esc_html__( 'Custom CSS Class', 'arabesque-core' ),
							'description' => esc_html__( 'Style particular content element differently - add a class name and refer to it in custom CSS', 'arabesque-core' )
						),
						array(
							'type'        => 'attach_image',
							'param_name'  => 'image',
							'heading'     => esc_html__( 'Image', 'arabesque-core' ),
							'description' => esc_html__( 'Select image from media library', 'arabesque-core' )
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'image_size',
							'heading'     => esc_html__( 'Image Size', 'arabesque-core' ),
							'description' => esc_html__( 'Enter image size. Example: thumbnail, medium, large, full or other sizes defined by current theme. Alternatively enter image size in pixels: 200x100 (Width x Height). Leave empty to use "thumbnail" size', 'arabesque-core' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'enable_image_shadow',
							'heading'     => esc_html__( 'Enable Image Shadow', 'arabesque-core' ),
							'value'       => array_flip( arabesque_mikado_get_yes_no_select_array( false ) ),
							'save_always' => true
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'image_behavior',
							'heading'    => esc_html__( 'Image Behavior', 'arabesque-core' ),
							'value'      => array(
								esc_html__( 'None', 'arabesque-core' )             => '',
								esc_html__( 'Open Lightbox', 'arabesque-core' )    => 'lightbox',
								esc_html__( 'Open Custom Link', 'arabesque-core' ) => 'custom-link',
								esc_html__( 'Zoom', 'arabesque-core' )             => 'zoom',
								esc_html__( 'Grayscale', 'arabesque-core' )        => 'grayscale'
							)
						),
						array(
							'type'       => 'textfield',
							'param_name' => 'custom_link',
							'heading'    => esc_html__( 'Custom Link', 'arabesque-core' ),
							'dependency' => Array( 'element' => 'image_behavior', 'value' => array( 'custom-link' ) )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'custom_link_target',
							'heading'    => esc_html__( 'Custom Link Target', 'arabesque-core' ),
							'value'      => array_flip( arabesque_mikado_get_link_target_array() ),
							'dependency' => Array( 'element' => 'image_behavior', 'value' => array( 'custom-link' ) )
						),
                        array(
                            'type'       => 'dropdown',
                            'param_name' => 'image_hover_animation',
                            'heading'    => esc_html__( 'Image Hover Animation', 'arabesque-core' ),
                            'value'      => array(
                                esc_html__( 'Moving Image', 'arabesque-core' )    => 'moving-image',
                                esc_html__( 'Box Behind', 'arabesque-core' )      => 'box-behind'
                            ),
                            'dependency' => Array( 'element' => 'image_behavior', 'value' => array( 'custom-link' ) )
                        ),
                        array(
                            'type'       => 'dropdown',
                            'param_name' => 'animation',
                            'heading'    => esc_html__( 'Appear Animation', 'arabesque-core' ),
                            'value'      => array_flip( arabesque_mikado_get_yes_no_select_array( false, false ) ),
                        ),
                        array(
                            'type'       => 'textfield',
                            'param_name' => 'animation_delay',
                            'heading'    => esc_html__( 'Animation Delay (ms)', 'arabesque-core' ),
                            'dependency' => array( 'element' => 'animation', 'value' => 'yes' )
                        ),
						array(
							'type'       => 'textfield',
							'param_name' => 'title',
							'heading'    => esc_html__( 'Title', 'arabesque-core' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'title_tag',
							'heading'     => esc_html__( 'Title Tag', 'arabesque-core' ),
							'value'       => array_flip( arabesque_mikado_get_title_tag( true, array( 'span' => 'span' ) ) ),
							'save_always' => true,
							'dependency'  => array( 'element' => 'title', 'not_empty' => true )
						),
                        array(
                            'type'       => 'textfield',
                            'param_name' => 'font_family',
                            'heading'    => esc_html__( 'Title Font Family', 'arabesque-core' ),
                            'description' => esc_html__( 'If not set, default heading font will be used', 'arabesque-core' ),
                            'save_always' => true,
                            'dependency'  => array( 'element' => 'title', 'not_empty' => true )
                        ),
                        array(
                            'type'       => 'textfield',
                            'param_name' => 'font_size',
                            'heading'    => esc_html__( 'Title Font Size (px or em)', 'arabesque-core' ),
                            'dependency'  => array( 'element' => 'title', 'not_empty' => true )
                        ),
                        array(
                            'type'       => 'textfield',
                            'param_name' => 'line_height',
                            'heading'    => esc_html__( 'Title Line Height (px or em)', 'arabesque-core' ),
                            'dependency'  => array( 'element' => 'title', 'not_empty' => true )
                        ),
                        array(
                            'type'        => 'dropdown',
                            'param_name'  => 'font_weight',
                            'heading'     => esc_html__( 'Title Font Weight', 'arabesque-core' ),
                            'value'       => array_flip( arabesque_mikado_get_font_weight_array( true ) ),
                            'save_always' => true,
                            'dependency'  => array( 'element' => 'title', 'not_empty' => true )
                        ),
                        array(
                            'type'        => 'dropdown',
                            'param_name'  => 'font_style',
                            'heading'     => esc_html__( 'Title Font Style', 'arabesque-core' ),
                            'value'       => array_flip( arabesque_mikado_get_font_style_array( true ) ),
                            'save_always' => true,
                            'dependency'  => array( 'element' => 'title', 'not_empty' => true )
                        ),
                        array(
                            'type'       => 'textfield',
                            'param_name' => 'letter_spacing',
                            'heading'    => esc_html__( 'Title Letter Spacing (px or em)', 'arabesque-core' ),
                            'dependency'  => array( 'element' => 'title', 'not_empty' => true )
                        ),
                        array(
                            'type'        => 'dropdown',
                            'param_name'  => 'text_transform',
                            'heading'     => esc_html__( 'Title Text Transform', 'arabesque-core' ),
                            'value'       => array_flip( arabesque_mikado_get_text_transform_array( true ) ),
                            'save_always' => true,
                            'dependency'  => array( 'element' => 'title', 'not_empty' => true )
                        ),
                        array(
                            'type'        => 'dropdown',
                            'param_name'  => 'text_decoration',
                            'heading'     => esc_html__( 'Title Text Decoration', 'arabesque-core' ),
                            'value'       => array_flip( arabesque_mikado_get_text_decorations( true ) ),
                            'save_always' => true,
                            'dependency'  => array( 'element' => 'title', 'not_empty' => true )
                        ),
						array(
							'type'       => 'colorpicker',
							'param_name' => 'title_color',
							'heading'    => esc_html__( 'Title Color', 'arabesque-core' ),
							'dependency' => array( 'element' => 'title', 'not_empty' => true )
						),
						array(
							'type'       => 'textfield',
							'param_name' => 'title_top_margin',
							'heading'    => esc_html__( 'Title Top Margin (px)', 'arabesque-core' ),
							'dependency' => array( 'element' => 'title', 'not_empty' => true )
						),
						array(
							'type'       => 'textarea',
							'param_name' => 'text',
							'heading'    => esc_html__( 'Text', 'arabesque-core' )
						),
						array(
							'type'       => 'colorpicker',
							'param_name' => 'text_color',
							'heading'    => esc_html__( 'Text Color', 'arabesque-core' ),
							'dependency' => array( 'element' => 'text', 'not_empty' => true )
						),
						array(
							'type'       => 'textfield',
							'param_name' => 'text_top_margin',
							'heading'    => esc_html__( 'Text Top Margin (px)', 'arabesque-core' ),
							'dependency' => array( 'element' => 'text', 'not_empty' => true )
						)
					)
				)
			);
		}
	}
	
	public function render( $atts, $content = null ) {
		$args   = array(
			'custom_class'           => '',
			'image'                  => '',
			'image_size'             => 'full',
			'enable_image_shadow'    => 'no',
			'image_behavior'         => '',
			'custom_link'            => '',
			'custom_link_target'     => '_self',
            'image_hover_animation'  => 'moving-image',
            'animation'              => '',
            'animation_delay'        => '',
			'title'                  => '',
			'title_tag'              => 'h5',
            'font_family'            => '',
            'font_size'              => '',
            'line_height'            => '',
            'font_weight'            => '',
            'font_style'             => '',
            'letter_spacing'         => '',
            'text_transform'         => '',
            'text_decoration'        => '',
			'title_color'            => '',
			'title_top_margin'       => '',
			'text'                   => '',
			'text_color'             => '',
			'text_top_margin'        => ''
		);
		$params = shortcode_atts( $args, $atts );
		
		$params['holder_classes']     = $this->getHolderClasses( $params );
        $params['holder_data']        = $this->getHolderData( $params );
		$params['image']              = $this->getImage( $params );
		$params['image_size']         = $this->getImageSize( $params['image_size'] );
		$params['image_behavior']     = ! empty( $params['image_behavior'] ) ? $params['image_behavior'] : $args['image_behavior'];
		$params['image_hover_animation'] = ! empty( $params['image_hover_animation'] ) ? $params['image_hover_animation'] : $args['image_hover_animation'];
		$params['custom_link_target'] = ! empty( $params['custom_link_target'] ) ? $params['custom_link_target'] : $args['custom_link_target'];
		$params['title_tag']          = ! empty( $params['title_tag'] ) ? $params['title_tag'] : $args['title_tag'];
		$params['title_styles']       = $this->getTitleStyles( $params );
		$params['text_styles']        = $this->getTextStyles( $params );
		
		$html = arabesque_core_get_shortcode_module_template_part( 'templates/image-with-text', 'image-with-text', '', $params );
		
		return $html;
	}
	
	private function getHolderClasses( $params ) {
		$holderClasses = array();
		
		$holderClasses[] = ! empty( $params['custom_class'] ) ? esc_attr( $params['custom_class'] ) : '';
		$holderClasses[] = $params['enable_image_shadow'] === 'yes' ? 'mkdf-has-shadow' : '';
		$holderClasses[] = ! empty( $params['image_behavior'] ) ? 'mkdf-image-behavior-' . $params['image_behavior'] : '';
		$holderClasses[] = ! empty( $params['image_hover_animation'] ) ? 'mkdf-image-hover-animation-' . $params['image_hover_animation'] : '';
        $holderClasses[] = ! empty( $params['animation'] ) ? 'mkdf-iwt-appear-animation-' . $params['animation'] : '';
		
		return implode( ' ', $holderClasses );
	}

    private function getHolderData( $params ) {
        $HolderData = array();

        if ( ( isset( $params['animation'] ) && $params['animation'] == 'yes' )
            && ( isset( $params['animation_delay'] ) && $params['animation_delay'] !== '' )
        ) {
            $HolderData['data-animation-delay'] = $params['animation_delay'];
        }

        return $HolderData;
    }
	
	private function getImage( $params ) {
		$image = array();
		
		if ( ! empty( $params['image'] ) ) {
			$id = $params['image'];
			
			$image['image_id'] = $id;
			$image_original    = wp_get_attachment_image_src( $id, 'full' );
			$image['url']      = $image_original[0];
			$image['alt']      = get_post_meta( $id, '_wp_attachment_image_alt', true );
		}
		
		return $image;
	}
	
	private function getImageSize( $image_size ) {
		$image_size = trim( $image_size );
		//Find digits
		preg_match_all( '/\d+/', $image_size, $matches );
		if ( in_array( $image_size, array( 'thumbnail', 'thumb', 'medium', 'large', 'full' ) ) ) {
			return $image_size;
		} elseif ( ! empty( $matches[0] ) ) {
			return array(
				$matches[0][0],
				$matches[0][1]
			);
		} else {
			return 'thumbnail';
		}
	}
	
	private function getTitleStyles( $params ) {
		$styles = array();
		
		if ( ! empty( $params['title_color'] ) ) {
			$styles[] = 'color: ' . $params['title_color'];
		}
		
		if ( $params['title_top_margin'] !== '' ) {
			$styles[] = 'margin-top: ' . arabesque_mikado_filter_px( $params['title_top_margin'] ) . 'px';
		}

        if ( $params['font_family'] !== '' ) {
            $styles[] = 'font-family: ' . $params['font_family'];
        }

        if ( ! empty( $params['font_size'] ) ) {
            if ( arabesque_mikado_string_ends_with( $params['font_size'], 'px' ) || arabesque_mikado_string_ends_with( $params['font_size'], 'em' ) ) {
                $styles[] = 'font-size: ' . $params['font_size'];
            } else {
                $styles[] = 'font-size: ' . $params['font_size'] . 'px';
            }
        }

        if ( ! empty( $params['line_height'] ) ) {
            if ( arabesque_mikado_string_ends_with( $params['line_height'], 'px' ) || arabesque_mikado_string_ends_with( $params['line_height'], 'em' ) ) {
                $styles[] = 'line-height: ' . $params['line_height'];
            } else {
                $styles[] = 'line-height: ' . $params['line_height'] . 'px';
            }
        }

        if ( ! empty( $params['font_weight'] ) ) {
            $styles[] = 'font-weight: ' . $params['font_weight'];
        }

        if ( ! empty( $params['font_style'] ) ) {
            $styles[] = 'font-style: ' . $params['font_style'];
        }

        if ( ! empty( $params['letter_spacing'] ) ) {
            if ( arabesque_mikado_string_ends_with( $params['letter_spacing'], 'px' ) || arabesque_mikado_string_ends_with( $params['letter_spacing'], 'em' ) ) {
                $styles[] = 'letter-spacing: ' . $params['letter_spacing'];
            } else {
                $styles[] = 'letter-spacing: ' . $params['letter_spacing'] . 'px';
            }
        }

        if ( ! empty( $params['text_transform'] ) ) {
            $styles[] = 'text-transform: ' . $params['text_transform'];
        }

        if ( ! empty( $params['text_decoration'] ) ) {
            $styles[] = 'text-decoration: ' . $params['text_decoration'];
        }
		
		return implode( ';', $styles );
	}
	
	private function getTextStyles( $params ) {
		$styles = array();
		
		if ( ! empty( $params['text_color'] ) ) {
			$styles[] = 'color: ' . $params['text_color'];
		}
		
		if ( $params['text_top_margin'] !== '' ) {
			$styles[] = 'margin-top: ' . arabesque_mikado_filter_px( $params['text_top_margin'] ) . 'px';
		}
		
		return implode( ';', $styles );
	}
}